/*
 * Copyright (C) 2000 - 2024 Silverpeas
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * As a special exception to the terms and conditions of version 3.0 of
 * the GPL, you may redistribute this Program in connection with Free/Libre
 * Open Source Software ("FLOSS") applications as described in Silverpeas's
 * FLOSS exception.  You should have received a copy of the text describing
 * the FLOSS exception, and it is also available here:
 * "https://www.silverpeas.org/legal/floss_exception.html"
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.silverpeas.core.contribution.model;

import org.silverpeas.core.Identifiable;

/**
 * A contribution with a single content that is managed by an application of the Silverpeas
 * collaborative portal. The content of the contribution can be empty. This interface defines the
 * common properties the different type of content in Silverpeas has to support. It is defined
 * for compatibility reason with the old content API (in which the contribution and content concepts
 * are entangled). This interface links the old Content Management API with the new one
 * represented by the {@link Contribution} and {@link ContributionContent} models.
 */
public interface SilverpeasContent extends Identifiable, Contribution {

  /**
   * Gets the identifier of this content in the Silverpeas component providing it. This identifier
   * is only unique among all the contents managed by the same component (whatever its different
   * instances). As each type of contents in Silverpeas is provided by a single Silverpeas
   * component, the identifier of a content is then specific to the component it belongs to. It is a
   * way for an instance of a such component to identify uniquely the different contents it manages.
   * So, each component can have their own policy to identify their content, whatever the way they
   * are identified in Silverpeas.
   * @return the identifier of this content.
   */
  String getId();

  /**
   * Gets the unique identifier of the Silverpeas component instance that manages this content.
   * @return the unique identifier of the component instance in the Silverpeas collaborative portal.
   */
  String getComponentInstanceId();

  /**
   * Gets the unique identifier of this content among all the contents managed in the Silverpeas
   * collaborative portal. It is the alone unique identifier of a content in the whole Silverpeas
   * portal; it is referred as the Silverpeas content identifier or the silver content identifier.
   * For each content put into the Silverpeas collaborative portal, an entry is uniquely created in
   * the whole system so that is can be referred by transversal services and by component instances
   * others the one that manages it. For compatibility reason, the Silverpeas content identifier of
   * contents that are no yet taken into account in the whole system isn't defined, so an empty
   * string is then returned.
   * @return the unique identifier of this content in the whole Silverpeas collaborative portal. Can
   * be empty if no such identifier is defined for the type of this content (default).
   */
  default String getSilverpeasContentId() {
    return "";
  }

  /**
   * {@link Contribution} default implementations.
   */

  @Override
  default ContributionIdentifier getIdentifier() {
    return ContributionIdentifier.from(getComponentInstanceId(), getId(), getContributionType());
  }
}
